import { NextResponse } from "next/server";
import { Resend } from "resend";

const SERVICE_LABELS: Record<string, string> = {
    fullstack: "Fullstack Systems",
    mobile: "Mobile Development",
    web: "Web Development",
    maintenance: "Maintenance",
    other: "Other",
};

export async function POST(request: Request) {
    try {
        const body = await request.json();
        const { name, email, service, message } = body;

        if (!name || !email || !message) {
            return NextResponse.json(
                { error: "Name, email, and message are required." },
                { status: 400 }
            );
        }

        const toEmail = process.env.CONTACT_EMAIL || "hello@sanvya.tech";
        const fromEmail = process.env.FROM_EMAIL || "Sanvya Contact <onboarding@resend.dev>";

        if (!process.env.RESEND_API_KEY) {
            console.error("RESEND_API_KEY is not set");
            return NextResponse.json(
                { error: "Email service is not configured." },
                { status: 500 }
            );
        }

        const serviceLabel = service ? SERVICE_LABELS[service] || service : "Not specified";
        const resend = new Resend(process.env.RESEND_API_KEY);

        const { error } = await resend.emails.send({
            from: fromEmail,
            to: [toEmail],
            subject: `Contact form: ${name} — ${serviceLabel}`,
            html: `
                <h2>New contact form submission</h2>
                <p><strong>Name:</strong> ${escapeHtml(name)}</p>
                <p><strong>Email:</strong> ${escapeHtml(email)}</p>
                <p><strong>Service interest:</strong> ${escapeHtml(serviceLabel)}</p>
                <p><strong>Message:</strong></p>
                <pre style="white-space: pre-wrap; font-family: inherit;">${escapeHtml(message)}</pre>
            `,
        });

        if (error) {
            console.error("Resend error:", error);
            return NextResponse.json(
                { error: "Failed to send email." },
                { status: 500 }
            );
        }

        return NextResponse.json({ success: true });
    } catch (err) {
        console.error("Contact API error:", err);
        return NextResponse.json(
            { error: "Something went wrong." },
            { status: 500 }
        );
    }
}

function escapeHtml(text: string): string {
    const map: Record<string, string> = {
        "&": "&amp;",
        "<": "&lt;",
        ">": "&gt;",
        '"': "&quot;",
        "'": "&#039;",
    };
    return text.replace(/[&<>"']/g, (c) => map[c] ?? c);
}
