"use client";

import Navbar from "@/components/Navbar";
import EndingSection from "@/components/EndingSection";
import ServiceDropdown from "@/components/ServiceDropdown";
import { useState } from "react";

type SubmitStatus = "idle" | "loading" | "success" | "error";

export default function ContactPage() {
    const [formData, setFormData] = useState({
        name: "",
        email: "",
        service: "",
        message: "",
    });
    const [status, setStatus] = useState<SubmitStatus>("idle");
    const [statusMessage, setStatusMessage] = useState("");

    const handleChange = (e: React.ChangeEvent<HTMLInputElement | HTMLTextAreaElement | HTMLSelectElement>) => {
        setFormData({ ...formData, [e.target.name]: e.target.value });
        if (status !== "idle") setStatus("idle");
    };

    const handleSubmit = async (e: React.FormEvent) => {
        e.preventDefault();
        setStatus("loading");
        setStatusMessage("");

        try {
            const res = await fetch("/api/contact", {
                method: "POST",
                headers: { "Content-Type": "application/json" },
                body: JSON.stringify(formData),
            });
            const data = await res.json().catch(() => ({}));

            if (!res.ok) {
                setStatus("error");
                setStatusMessage(data.error || "Something went wrong. Please try again.");
                return;
            }
            setStatus("success");
            setStatusMessage("Thanks for reaching out! We will get back to you shortly.");
            setFormData({ name: "", email: "", service: "", message: "" });
        } catch {
            setStatus("error");
            setStatusMessage("Network error. Please try again.");
        }
    };

    return (
        <main style={{ background: "var(--bg-page)", minHeight: "100vh" }}>
            {/* NAVBAR */}
            <Navbar />

            {/* CONTACT HERO SECTION */}
            <section
                className="section"
                style={{
                    paddingTop: "calc(var(--space-5xl) + 64px)",
                    paddingBottom: "var(--space-5xl)",
                    minHeight: "80vh",
                    display: "flex",
                    flexDirection: "column",
                    justifyContent: "center",
                }}
            >
                <div className="container" style={{ textAlign: "center" }}>
                    <h1
                        className="display"
                        style={{
                            marginBottom: "var(--space-2xl)",
                            maxWidth: "20ch",
                            margin: "0 auto var(--space-2xl) auto",
                        }}
                    >
                        Let's Work Together
                    </h1>

                    <p
                        className="body-lg"
                        style={{
                            maxWidth: "40ch",
                            margin: "0 auto var(--space-4xl)",
                            opacity: 0.85,
                        }}
                    >
                        Have a project in mind? We'd love to hear about it. Reach out and let's create something exceptional together.
                    </p>

                    {/* Contact Form */}
                    <div
                        style={{
                            maxWidth: "600px",
                            margin: "0 auto",
                            textAlign: "left",
                        }}
                    >
                        <form onSubmit={handleSubmit} style={{ display: "flex", flexDirection: "column", gap: "var(--space-lg)" }}>
                            <div>
                                <label htmlFor="name" style={{ display: "block", marginBottom: "var(--space-xs)", fontWeight: 500 }}>Name</label>
                                <input
                                    type="text"
                                    id="name"
                                    name="name"
                                    required
                                    value={formData.name}
                                    onChange={handleChange}
                                    style={{
                                        width: "100%",
                                        padding: "var(--space-md)",
                                        borderRadius: "var(--radius-md)",
                                        border: "1px solid var(--border-subtle)",
                                        background: "var(--bg-surface)",
                                        color: "var(--text-primary)",
                                        fontFamily: "inherit",
                                        fontSize: "1rem",
                                    }}
                                />
                            </div>

                            <div>
                                <label htmlFor="email" style={{ display: "block", marginBottom: "var(--space-xs)", fontWeight: 500 }}>Email</label>
                                <input
                                    type="email"
                                    id="email"
                                    name="email"
                                    required
                                    value={formData.email}
                                    onChange={handleChange}
                                    style={{
                                        width: "100%",
                                        padding: "var(--space-md)",
                                        borderRadius: "var(--radius-md)",
                                        border: "1px solid var(--border-subtle)",
                                        background: "var(--bg-surface)",
                                        color: "var(--text-primary)",
                                        fontFamily: "inherit",
                                        fontSize: "1rem",
                                    }}
                                />
                            </div>

                            <div>
                                <label htmlFor="service" style={{ display: "block", marginBottom: "var(--space-xs)", fontWeight: 500 }}>Service Interest</label>
                                <ServiceDropdown
                                    id="service"
                                    value={formData.service}
                                    onChange={(value) => setFormData(prev => ({ ...prev, service: value }))}
                                    options={[
                                        { label: "Fullstack Systems", value: "fullstack" },
                                        { label: "Mobile Development", value: "mobile" },
                                        { label: "Web Development", value: "web" },
                                        { label: "Maintenance", value: "maintenance" },
                                        { label: "Other", value: "other" },
                                    ]}
                                />
                            </div>

                            <div>
                                <label htmlFor="message" style={{ display: "block", marginBottom: "var(--space-xs)", fontWeight: 500 }}>Message</label>
                                <textarea
                                    id="message"
                                    name="message"
                                    required
                                    rows={5}
                                    value={formData.message}
                                    onChange={handleChange}
                                    style={{
                                        width: "100%",
                                        padding: "var(--space-md)",
                                        borderRadius: "var(--radius-md)",
                                        border: "1px solid var(--border-subtle)",
                                        background: "var(--bg-surface)",
                                        color: "var(--text-primary)",
                                        fontFamily: "inherit",
                                        fontSize: "1rem",
                                        resize: "vertical",
                                    }}
                                />
                            </div>

                            {statusMessage && (
                                <p
                                    style={{
                                        fontSize: "0.9rem",
                                        padding: "var(--space-md)",
                                        borderRadius: "var(--radius-md)",
                                        background: status === "error" ? "rgba(239, 68, 68, 0.15)" : "rgba(34, 197, 94, 0.15)",
                                        color: status === "error" ? "var(--text-primary)" : "var(--text-primary)",
                                    }}
                                >
                                    {statusMessage}
                                </p>
                            )}
                            <button
                                type="submit"
                                className="btn-primary"
                                disabled={status === "loading"}
                                style={{
                                    marginTop: "var(--space-md)",
                                    width: "100%",
                                    justifyContent: "center",
                                    opacity: status === "loading" ? 0.7 : 1,
                                    cursor: status === "loading" ? "not-allowed" : "pointer",
                                }}
                            >
                                {status === "loading" ? "Sending…" : "Send Message"}
                            </button>
                        </form>
                    </div>

                    {/* Email Link */}
                    <div
                        style={{
                            marginTop: "var(--space-4xl)",
                            textAlign: "center",
                        }}
                    >
                        <p className="body" style={{ opacity: 0.7, marginBottom: "var(--space-sm)" }}>Or email us directly at</p>
                        <a
                            href="mailto:hello@sanvya.tech"
                            style={{
                                display: "inline-block",
                                fontSize: "1.125rem",
                                fontWeight: 500,
                                color: "var(--text-primary)",
                                textDecoration: "none",
                            }}
                        >
                            hello@sanvya.tech
                        </a>
                    </div>
                </div>
            </section >

            {/* ENDING SECTION */}
            < EndingSection />
        </main >
    );
}
