"use client";

import { useRef } from "react";
import gsap from "gsap";
import { ScrollTrigger } from "gsap/ScrollTrigger";
import { useGSAP } from "@gsap/react";

gsap.registerPlugin(ScrollTrigger, useGSAP);

interface SectionProps {
    children?: React.ReactNode;
    className?: string;
    variant?: "default" | "media-left" | "media-right" | "centered";
    mediaUrl?: string;
    mediaAspect?: "16/9" | "21/9" | "4/3" | "1/1";
    heading?: string;
    body?: string;
}

export default function Section({
    children,
    className = "",
    variant = "default",
    mediaUrl,
    mediaAspect = "16/9",
    heading,
    body,
}: SectionProps) {
    const sectionRef = useRef<HTMLElement>(null);
    const fadeRef = useRef<HTMLDivElement>(null);

    useGSAP(
        () => {
            const el = fadeRef.current;
            if (!el) return;

            gsap.fromTo(
                el,
                {
                    opacity: 0,
                    y: 40,
                },
                {
                    opacity: 1,
                    y: 0,
                    duration: 1.2,
                    ease: "power2.out",
                    scrollTrigger: {
                        trigger: sectionRef.current,
                        start: "top 80%",
                        toggleActions: "play none none reverse",
                    },
                }
            );
        },
        { scope: sectionRef }
    );

    const aspectClass = {
        "16/9": "aspect-16-9",
        "21/9": "aspect-21-9",
        "4/3": "aspect-4-3",
        "1/1": "aspect-1-1",
    }[mediaAspect];

    // Media container component
    const MediaContainer = () => (
        <div className={`media-container ${aspectClass} placeholder-media`}>
            {mediaUrl ? (
                <video autoPlay muted loop playsInline>
                    <source src={mediaUrl} type="video/mp4" />
                </video>
            ) : (
                <span>Media Placeholder</span>
            )}
        </div>
    );

    // Text content component
    const TextContent = () => (
        <div>
            {heading && <h2 className="heading-lg" style={{ marginBottom: "var(--space-lg)" }}>{heading}</h2>}
            {body && <p className="body-lg">{body}</p>}
            {children}
        </div>
    );

    // Render based on variant
    const renderContent = () => {
        if (variant === "centered") {
            return (
                <div style={{ maxWidth: "800px", margin: "0 auto", textAlign: "center" }}>
                    <TextContent />
                </div>
            );
        }

        if (variant === "media-left" || variant === "media-right") {
            return (
                <div
                    style={{
                        display: "grid",
                        gridTemplateColumns: "1fr",
                        gap: "var(--space-2xl)",
                        alignItems: "center",
                    }}
                    className="media-grid"
                >
                    {variant === "media-left" ? (
                        <>
                            <MediaContainer />
                            <TextContent />
                        </>
                    ) : (
                        <>
                            <TextContent />
                            <MediaContainer />
                        </>
                    )}
                </div>
            );
        }

        return <TextContent />;
    };

    return (
        <section ref={sectionRef} className={`section ${className}`} style={{ position: "relative", zIndex: 1 }}>
            <div ref={fadeRef} className="container">
                {renderContent()}
            </div>
        </section>
    );
}


