"use client";

import { useRef } from "react";
import gsap from "gsap";
import { ScrollTrigger } from "gsap/ScrollTrigger";
import { useGSAP } from "@gsap/react";

gsap.registerPlugin(ScrollTrigger, useGSAP);

interface VideoSectionProps {
    videoUrl?: string;
    aspectRatio?: "16/9" | "21/9" | "4/3";
    placeholder?: string;
}

export default function VideoSection({
    videoUrl,
    aspectRatio = "21/9",
    placeholder = "Contextual Video Background",
}: VideoSectionProps) {
    const sectionRef = useRef<HTMLElement>(null);
    const videoContainerRef = useRef<HTMLDivElement>(null);

    const aspectClass = {
        "16/9": "aspect-16-9",
        "21/9": "aspect-21-9",
        "4/3": "aspect-4-3",
    }[aspectRatio];

    useGSAP(
        () => {
            const container = videoContainerRef.current;
            if (!container) return;

            // Fade in animation when entering viewport with scale and blur
            const fadeInTl = gsap.timeline({
                scrollTrigger: {
                    trigger: sectionRef.current,
                    start: "top 75%",
                    end: "top 25%",
                    toggleActions: "play none none none",
                },
            });

            fadeInTl.fromTo(
                container,
                {
                    opacity: 0,
                    y: 80,
                    scale: 0.92,
                    filter: "blur(12px)",
                },
                {
                    opacity: 1,
                    y: 0,
                    scale: 1,
                    filter: "blur(0px)",
                    duration: 1.6,
                    ease: "power2.out",
                }
            );

            // Fade out animation when scrolling DOWN past the section
            const fadeOutTl = gsap.timeline({
                scrollTrigger: {
                    trigger: sectionRef.current,
                    start: "bottom 60%",
                    end: "bottom 30%",
                    scrub: 1.2,
                    onLeaveBack: () => {
                        // Reset to visible state when scrolling back up
                        gsap.set(container, {
                            opacity: 1,
                            y: 0,
                            scale: 1,
                            filter: "blur(0px)",
                        });
                    },
                },
            });

            fadeOutTl.to(container, {
                opacity: 0,
                y: -60,
                scale: 0.88,
                filter: "blur(16px)",
            });
        },
        { scope: sectionRef }
    );

    return (
        <section
            ref={sectionRef}
            className="section video-section-full"
            style={{ paddingTop: "var(--space-5xl)", paddingBottom: "var(--space-5xl)" }}
        >
            <div
                ref={videoContainerRef}
                className={`media-container ${aspectClass} ${!videoUrl ? "placeholder-media" : ""}`}
            >
                {videoUrl ? (
                    <>
                        {/* Main crisp video */}
                        <video
                            className="main-video"
                            autoPlay
                            muted
                            loop
                            playsInline
                            style={{ position: 'relative', zIndex: 10 }}
                        >
                            <source src={videoUrl} type="video/mp4" />
                        </video>

                        {/* Blurred ambient background video */}
                        <video
                            className="blur-video"
                            autoPlay
                            muted
                            loop
                            playsInline
                            style={{
                                position: 'absolute',
                                top: 0,
                                left: 0,
                                width: '100%',
                                height: '100%',
                                objectFit: 'cover',
                                filter: 'blur(40px) saturate(1.5)',
                                transform: 'scale(1.1)',
                                opacity: 0.6,
                                zIndex: 0,
                                pointerEvents: 'none'
                            }}
                        >
                            <source src={videoUrl} type="video/mp4" />
                        </video>
                    </>
                ) : (
                    <span>{placeholder}</span>
                )}
            </div>
        </section>
    );
}
